/**
 * Select autocomplete
 *
 * @mixin
 *
 * Handles logic when using the "autocomplete" prop
 */
export default {
  props: {
    filter: {
      type: Function,
      default: function _default(item, queryText, itemText) {
        var hasValue = function hasValue(val) {
          return [undefined, null].includes(val);
        };

        var text = hasValue(itemText) ? '' : itemText;
        var query = hasValue(queryText) ? '' : queryText;

        return text.toString().toLowerCase().indexOf(query.toString().toLowerCase()) > -1;
      }
    }
  },

  methods: {
    genFiltered: function genFiltered(text) {
      if (!this.isAutocomplete || !this.searchValue || this.filteredItems.length < 1) return text;

      text = (text || '').toString();

      var _getMaskedCharacters = this.getMaskedCharacters(text),
          start = _getMaskedCharacters.start,
          middle = _getMaskedCharacters.middle,
          end = _getMaskedCharacters.end;

      return '' + start + this.genHighlight(middle) + end;
    },
    genHighlight: function genHighlight(text) {
      if (this.isNotFiltering) return text;

      return '<span class="list__tile__mask">' + text + '</span>';
    },
    getMaskedCharacters: function getMaskedCharacters(text) {
      var searchValue = (this.searchValue || '').toString().toLowerCase();
      var index = text.toLowerCase().indexOf(searchValue);

      if (index < 0) return { start: '', middle: text, end: '' };

      var start = text.slice(0, index);
      var middle = text.slice(index, index + searchValue.length);
      var end = text.slice(index + searchValue.length);
      return { start: start, middle: middle, end: end };
    },
    filterSearch: function filterSearch() {
      var _this = this;

      if (!this.isAutocomplete) return this.computedItems;

      return this.computedItems.filter(function (i) {
        return _this.filter(i, _this.searchValue, _this.getText(i));
      });
    },
    onKeyDown: function onKeyDown(e) {
      var _this2 = this;

      // If enter, space, up, or down is pressed, open menu
      if (!this.menuIsActive && [13, 32, 38, 40].includes(e.keyCode)) {
        e.preventDefault();
        return this.showMenuItems();
      }

      // If escape or tab with no search, blur
      if (e.keyCode === 27 || e.keyCode === 9 && !this.searchValue) {
        return this.blur();
      }

      // Tab shouldn't switch inputs
      if (e.keyCode === 9) {
        this.tags ? e.preventDefault() : this.blur();
      }

      if (!this.isAutocomplete || ![32].includes(e.keyCode) // space
      ) this.$refs.menu.changeListIndex(e);

      // Up or down
      if ([38, 40].includes(e.keyCode)) this.selectedIndex = -1;

      if (this.isAutocomplete && !this.hideSelections && !this.searchValue) this.changeSelectedIndex(e.keyCode);

      if (!this.tags || !this.searchValue) return;

      // Tab, enter
      if ([9, 13].includes(e.keyCode)) {
        var newItem = this.filteredItems.length && this.$refs.menu.listIndex >= 0 ? this.filteredItems[this.$refs.menu.listIndex] : this.searchValue;
        this.addTag(newItem);
      }

      // Left arrow
      if (e.keyCode === 37 && this.$refs.input.selectionStart === 0 && this.selectedItems.length) {
        this.addTag(this.searchValue);
        this.$nextTick(function () {
          _this2.selectedIndex = Math.max(_this2.selectedItems.length - 2, 0);
        });
      }

      // Right arrow
      if (e.keyCode === 39 && this.$refs.input.selectionEnd === this.searchValue.length) {
        this.$refs.menu.listIndex = -1;
      }
    },
    addTag: function addTag(content) {
      var _this3 = this;

      if (this.selectedItems.includes(content)) {
        this.$delete(this.selectedItems, this.selectedItems.indexOf(content));
      } else {
        this.selectedItems.push(content);
      }
      this.$nextTick(function () {
        _this3.searchValue = null;
        _this3.$emit('change', _this3.selectedItems);
      });
    }
  }
};