# Form Radio Inputs

> For cross browser consistency, `<b-form-radio-group>` and `<b-form-radio>` uses
Bootstrap's custom radio input to replace the browser default radio input. It
is built on top of semantic and accessible markup, so it is a solid replacement
for the default radio input.

<div class="alert alert-danger">
  <h5>Breaking Change notice:</h5>
  <p class="m-0">
    Starting with release <samp>v1.0.0-beta.8</samp> of Bootstrap-Vue,
    <samp class="bg-transparent text-danger">&lt;b-form-radio&gt;</samp> now
    generates a <strong>single</strong> radio input. Please use
    <samp class="bg-transparent text-danger">&lt;b-form-radio-group&gt;</samp>
    to generate a series of radio inputs based on the
    <samp class="bg-transparent text-danger">options</samp> prop.
  </p>
</div>

The individual radio inputs in radio input group can be specified via the `options`
prop of `<b-form-radio-group>`, or via manual placement of the `<b-form-radio>` sub component.
  
```html
<template>
  <div>
    <h5>Radios using options</h5>
    <b-form-radio-group id="radios1" v-model="selected" :options="options" name="radioOpenions">
    </b-form-radio-group>

    <h5 class="mt-3">Radios using sub-component</h5>
    <b-form-radio-group id="radios2" v-model="selected" name="radioSubComponent">
      <b-form-radio value="first">Toggle this custom radio</b-form-radio>
      <b-form-radio value="second">Or toggle this other custom radio</b-form-radio>
      <b-form-radio value="third" disabled>This one is Disabled</b-form-radio>
      <b-form-radio :value="{fourth: 4}">This is the 4th radio</b-form-radio>
    </b-form-radio-group>

    <div class="mt-3">
      Selected: <strong>{{ selected }}</strong>
    </div>
  </div>
</template>

<script>
  export default {
    data: {
      selected: 'first',
      options: [
        { text: 'Toggle this custom radio', value: 'first' },
        { text: 'Or toggle this other custom radio', value: 'second' },
        { text: 'This one is Disabled', value: 'third', disabled: true },
        { text: 'This is the 4th radio', value: {fourth: 4} }
      ]
    }
  };
</script>

<!-- form-radio-1.vue -->
```
Feel free to mix and match `options` prop and `<b-form-radio>` in `<b-form-radio-group>`.
Manually placed `<b-form-radio>` inputs will appear _below_ any radio inputs generated by
the `options` prop. To have them apper _above_ the inputs generated by `options`, place them
in the named slot `first`.


## Options
Please see options in [`<b-form-select>`](/docs/components/form-select) docs for details on passing
options (value array) to `<b-form-radio-group>`.

## Size
Control the size of the radio text by setting the prop `size` to either `sm` for small or
`lg` for large.

```html
<template>
  <div>
    <h5>Small size radios</h5>
    <b-form-radio-group v-model="selected"
                        :options="options"
                        size="sm"
                        name="radiosSm">
    </b-form-radio-group>

    <br>
    <h5>Default size radios</h5>
    <b-form-radio-group v-model="selected"
                        :options="options"
                        name="radiosMd">
    </b-form-radio-group>
    
    <br>
    <h5>Large size radios</h5>
    <b-form-radio-group v-model="selected"
                        :options="options"
                        size="lg"
                        name="radiosLg">
    </b-form-radio-group>

    <div class="mt-4">
      Selected: <strong>{{ selected }}</strong>
    </div>
  </div>
</template>

<script>
  export default {
    data: {
      selected: 'first',
      options: [
        { text: 'First radio', value: 'first' },
        { text: 'Second radio', value: 'second' },
        { text: 'Third radio', value: 'third' }
      ]
    }
  }
</script>

<!-- form-radio-size-1.vue -->
```

**Note:** _the current Bootstrap V4.beta CSS does not correctly style the size of
the radio indicator._


## Inline or stacked
By default `<b-form-radio>` generates inline radio inputs. Set the prop `stacked` to make
the radios appear one over the other.

```html
<template>
  <div>
    <h5>Inline radios (default)</h5>
    <b-form-radio-group v-model="selected"
                        :options="options"
                        name="radioInline">
    </b-form-radio-group>

    <h5 class="mt-3">Stacked radios</h5>
    <b-form-radio-group v-model="selected"
                        :options="options"
                        stacked
                        name="radiosStacked">
    </b-form-radio-group>

    <div class="mt-3">
      Selected: <strong>{{ selected }}</strong>
    </div>
  </div>
</template>

<script>
  export default {
    data: {
      selected: 'first',
      options: [
        { text: 'First radio', value: 'first' },
        { text: 'Second radio', value: 'second' },
        { text: 'Third radio', value: 'third' }
      ]
    }
  }
</script>

<!-- form-radio-stacked-1.vue -->
```

## Button style radios
Render radios with the look of buttons by setting the prop `buttons`. Set the button variant by
setting the `button-variant` prop to one of the standard Bootstrap button variants (see
[`<b-button>`](./button) for supported variants). The default `button-variant` is `secondary`.

The `buttons` prop has precedence over `plain`, and `button-variant` has no effect if
`buttons` is not set.

Button style radios will have the class `.active` automatically applied to their label
when they are in the checked state.

```html
<template>
  <div>
    <h5>Button style radios</h5>
    <b-form-radio-group id="btnradios1"
                        buttons
                        v-model="selected"
                        :options="options"
                        name="radiosBtnDefault" />

    <h5 class="mt-3">
      Button style radios with <code>outline-primary</code> variant and size <code>lg</code>
    </h5>
    <b-form-radio-group id="btnradios2"
                        buttons
                        button-variant="outline-primary"
                        size="lg"
                        v-model="selected"
                        :options="options"
                        name="radioBtnOutline" />

    <h5 class="mt-3">Stacked button style radios</h5>
    <b-form-radio-group id="btnradios3"
                        buttons
                        stacked
                        v-model="selected"
                        :options="options"
                        name="radioBtnStacked" />
  </div>
</template>

<script>
export default {
  data: {
    selected: 'radio1',
    options: [
      { text: 'Radio 1', value: 'radio1' },
      { text: 'Radio 3', value: 'radio2' },
      { text: 'Radio 3 (disabled)', value: 'radio3', disabled: true },
      { text: 'Radio 4', value: 'radio4' }
    ]
  }
}
</script>

<!-- form-radio-buttons.vue -->
```

## Non custom style radio inputs (plain)
You can have `b-form-radio` render a browser native radio input by setting the `plain` prop.


```html
<template>
  <div>
    <h5>Plain inline radios</h5>
    <b-form-radio-group v-model="selected"
                        :options="options"
                        plain
                        name="plainInline" />

    <h5 class="mt-3">Plain stacked radios</h5>
    <b-form-radio-group v-model="selected"
                        :options="options"
                        plain
                        stacked
                        name="plainStacked" />

  </div>
</template>

<script>
  export default {
    data: {
      selected: 'first',
      options: [
        { text: 'First radio', value: 'first' },
        { text: 'Second radio', value: 'second' },
        { text: 'Third radio', value: 'third' }
      ]
    }
  }
</script>

<!-- form-radio-plain-1.vue -->
```

**Note:** `plain` will have no effect if `buttons` is set.


## Contextual States
Bootstrap includes validation styles for `valid` and `invalid` states
on most form controls.

Generally speaking, you’ll want to use a particular state for specific types of feedback:
- `'invalid'` is great for when there’s a blocking or required field. A user must fill in
this field properly to submit the form.
- `'valid'` is ideal for situations when you have per-field validation throughout a form
and want to encourage a user through the rest of the fields.
- `null` Displays no validation state

To apply one of the contextual state icons on `<b-form-radio>`, set the `state` prop
to `'invalid'` (or `false`), `'valid'` (or `true`), or `null`.

**Note:** contextual state is not supported for radios rendered in buttons mode.

### Conveying contextual validation state to assistive technologies and colorblind users:
Using these contextual states to denote the state of a form control only provides
a visual, color-based indication, which will not be conveyed to users of assistive
technologies - such as screen readers - or to colorblind users.

Ensure that an alternative indication of state is also provided. For instance, you
could include a hint about state in the form control's `<label>` text itself, or by
providing an additional help text block (i.e. `<b-form-feedbck>`). Specifically
for assistive technologies, invalid form controls can also be assigned an
`aria-invalid="true"` attribute (see below).


### ARIA `aria-invalid` attribute
When `<b-form-radio>` has an invalid contextual state (i.e. `invalid`) you may also
want to set the `<b-form-radio>` prop `aria-invalid` to `true`.

Supported `invalid` values are:
- `false` (default) No errors detected
- `true` The value has failed validation.

`aria-invalid` is automatically set if `state` is `invalid`.


## Radio component aliases
- `<b-form-radio-group>` can be used by the shorter aliant `<b-radio-group>`.
- `<b-form-radio>` can be used by the shorter alias of `<b-radio>`.

## Component Reference
